/** This file contains the functions for generating the set system
 * based on Alon's suggestion of geometric expanders. It is important
 * to note that using these expanders, our choices for "n" are
 * limited.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "primes.hpp"

// We work with a fixed dimension for the geoemetry viz. 2
#define Dimension 2

// Define an upper limit that l(n)=k can take :-
#define klimit 20

/* Points to note
   --------------
 * The input to the problem is "n", the length for which the expanding sets are desired
 * We develop a bipartile expander based on Alon's suggestion that involves using a finite geometry
   and its set of points and hyperplanes.
 * Every point is an array of size d+1 and so is every hyperplane.
 * We first discover the least value of q which satisfies 
 	 	n <= (q^(d+1) - 1)/(q - 1)
   and use this value of q for all computations. compute l(n) = k. compute n^prime (the rhs qty above).
 * How to generate the vertices of the graphs (i.e. the points (U) and the hy[erplanes (V) ) :-
 	 + We use a normalised set for both.
	 + Start with (0, 0, ..... 0, 1)
	 + At every step put the newly generated vertex into U (V). U (V) is an array of order n
	 + For all 0 <= i <= d+1, add all possible points (0, 0, .... 0, 1, x_(i+1)...x_(d+1)).
 * How to form edges
 	 + Work exhaustively..see whichever hyperplane fits a point..join the point to that 
	 	hyperplane. 
 * Return an n x l(n) array where each row is one l(n)-subset of [n].
*/

static int n; // n and k are common throughout the program.
static int m, k; // m is the smallest value >= n s.t. m = (p^(d+1) - 1)/(p - 1) for some prime p

// Function prototypes
void getGeometricChoices(int choiceOfN[], int maxChoice);
void generateGeometricExpander(int num, int *lnum, int subsets[][klimit]);
int getQ(int n);
void generateExpander(int q, int subsets[][klimit]);
void generateVertices(int q, int Vertices[][Dimension + 1]); 
void formEdges(int q, int Vertices[][Dimension + 1], int Edges[][klimit]);
void formSubsets(int Edges[][klimit], int Subsets[][klimit]); 
int findIncidence(int i, int j, int vertices[][Dimension + 1], int q);

// Function to return the first 'maxChoice' values for geometric 
// expanders possible
void getGeometricChoices(int choiceOfN[], int maxChoice) {
	 choiceOfN[0] = findN(2, Dimension);
	 int q1 = 3;
	 int count = 10;
	 for (int c=1; c<maxChoice; c++) {
		  choiceOfN[c] = findN(q1, Dimension);
		  q1 = getNextPrime(q1);
	 }
}

// Function to actually generate the expander into a set of subsets (subsets)
void generateGeometricExpander(int num, int *lnum, int subsets[][klimit]) {
	 n = num;
	 int q = getQ(n); // Will return the value of q that fits our requirement.
	 m = findN(q, Dimension);
	 k = findK(q, Dimension);	 
	 *lnum = k;
	 generateExpander(q, subsets);
}

// Function to select the q which fits the given n
int getQ(int n) {
	 int q = 2;
	 int previousQ;
	 int val = findN(q, Dimension);
	 if (n < val) {
	 	  printf("Value of n too small");
		  exit(1);
	 }
	 q = 3;
	 val = findN(q, Dimension);
	 if (n < val) {
	 	  return 2;
	 }
	 previousQ = 3;
	 do {
	 	  q = getNextPrime(previousQ);
	 	  val = findN(q, Dimension);
		  if (n < val) {
		  	    return previousQ;
		  }
		  previousQ = q;
	 } while(1);	   
}

// Function to generate the expander graph and return the subsets
void generateExpander(int q, int subsets[][klimit]) {
	 
	 int U[m][Dimension + 1]; // This is the set of vertices, U.
	 int Edges[m][klimit] ;	  // This is the adjacency matrix for the graph 

	 generateVertices(q, U); // This will form all vertices in U and V
	 formEdges(q, U, Edges);	// This will form the adjacency matrix
	 formSubsets(Edges, subsets); // This will form the desired subsets
}

// Function to form the vertices of both the points and the hyperplanes
void generateVertices(int q, int Vertices[][Dimension + 1]) {	 	
	 // We generate vertices in a normalised manner. First we generate (0, 0...0, 1)
	 // then the vertices of the form (0, 0....0, 1, x), then the q^2 vertices of
	 // the form (0, 0, ...0, 1, x, y) and so on.
	
	 // The first vertex (Vertices[0])
	 for (int l=0; l<Dimension; l++)
		  Vertices[0][l] = 0;
	 Vertices[0][Dimension] = 1;	 
		
	 int beginPrevious = 0; // Will be used for computing the set of q^(i+1) vertices
	 int endPrevious = 0;   // ---do----
	 int currentIndex = 0;
	 
	 for (int i=Dimension-1; i >= 0; i--) {
	 	  for (int j=beginPrevious; j<=endPrevious; j++) {
		  	   if (j == beginPrevious) {
			   		 beginPrevious = endPrevious + 1;
					 currentIndex = beginPrevious;
			   }
			   for (int ij=0; ij < q; ij++) {
			   		for (int ji=0; ji < i; ji++) 
					     Vertices[currentIndex][ji] = 0;
					Vertices[currentIndex][i] = 1;
					Vertices[currentIndex][i+1] = ij;
					for (int ji=i+2; ji <= Dimension; ji++) 
						 Vertices[currentIndex][ji] = Vertices[j][ji];
					currentIndex++;
			   }
		  }
		  endPrevious = currentIndex - 1;
	  }
}

// This function forms edges between the two sets of vertices (both represented by U in the call)
void formEdges(int q, int Vertices[][Dimension + 1], int Edges[][klimit]) {
	 int currentVertexNumber[m];
	 int dotProduct = 0;
	 for (int h=0; h<m; h++) 
	 	  currentVertexNumber[h] = 0;
	 
	 for (int i=0; i < m; i++) {
	 	  for (int j=i; j < m; j++) {
		  	   dotProduct = findIncidence(i, j, Vertices, q);
		  	   if (dotProduct == 0) {
			   		Edges[i][currentVertexNumber[i]] = j;
					Edges[j][currentVertexNumber[j]] = i;
					if (i == j) {
						currentVertexNumber[i]++;
					} else {
						currentVertexNumber[i]++;
						currentVertexNumber[j]++;
					}
			   }
		  }
	 }
}	

// Function to test incidence of a point on a plane
int findIncidence(int i, int j, int vertices[][Dimension + 1], int q) {
	 int sum = 0;
	 for (int index=0; index <= Dimension; index++) {
	 	  sum += (vertices[i][index] * vertices[j][index]) % q;
	 	  sum %= q;
	 }
	 return sum;
}

// This function gets the subsets from the set of edges formed above  
void formSubsets(int Edges[][klimit], int Subsets[][klimit]) {
	 for (int k11=0; k11<n; k11++) {
		  for (int k12=0; k12<k; k12++) {
			   Subsets[k11][k12] = Edges[k11][k12];
		  }
	 }
	 return;
}

