classdef AliasGHMM
    % define a 2-Aliased univariate Gaussian Hidden Markov Model
    
    properties
        n % number os states
        numComp % number of distinguished output components (for non-aliased numComp = n, for 2-aliased numComp = n-1)
        compWeights % component weights when estimated from mixture
        aliasComp % the component having two states, if 2-aliased, otherwise [].
        stateToComp % which of the numComp components is assigned to which state in 1:n
        A % n x n transition matrix
        mu % the vector of means for the Gaussian components
        sigma % the vector of variances for the Gaussian components
        initial % intial distribution over states
        kernel % the symmetric inner product matrix between the numComp components
        
        %%% inner properties %%%%%%%%
        Amin
        Amax
        Acum
        B
        smu  %  mu extended to a vector of length n with duplicate components as determined by stateToComp
        ssigma % sigma with duplicate components as determined by stateToComp
        stationary % stationary distribution
        initialCum
        pdfs % the output distribution's pdfs
        bayesPdfs % currently not in use
        mixture
    end
    
    methods
        %         function ghmm = NonAliasGHMM( n, A , mu, sigma)
        %             ghmm = NonAliasGHMM( n, A, mu, sigma, []);
        %         end
        
        
        
        function ghmm = AliasGHMM(n,A,sToC, mu,sigma, compWeights,initial, stationary)
            ghmm.n = n;
            ghmm.A = A;
            if ~isempty( A )
                ghmm.Acum = cumsum([zeros(1,n); A],1);
            else
                ghmm.Acum = [];
            end
            ghmm.mu = mu;
            ghmm.sigma = sigma;
            ghmm.compWeights = compWeights;
            if ~isempty( ghmm.compWeights )
                ghmm.numComp =   length( ghmm.compWeights );
            else
                ghmm.numComp = length( mu );
            end
            if isempty(stationary)
                ghmm.stationary =  ghmm.stationaryCalculate();
            else
                ghmm.stationary = stationary;
            end
            if isempty( initial )
                ghmm.initial =  ghmm.stationary;
            else
                ghmm.initial = initial;
            end
            if ~isempty( ghmm.initial )
                ghmm.initialCum = cumsum([0; ghmm.initial]);
            else
                ghmm.initialCum = [];
            end
            if ~isempty( ghmm.numComp )
                ghmm.pdfs = cell(1,ghmm.numComp );
                for i=1:ghmm.numComp
                    ghmm.pdfs{i} = @(x) normpdf(x,mu(i),sigma(i));
                end
            end
            ghmm.kernel = ghmm.kernelCalculate();
            ghmm.stateToComp = sToC;
            if ~isempty(  ghmm.stateToComp )
                ghmm = ghmm.setStateToComp(  sToC);
            else
                ghmm.stateToComp  = sToC;
                ghmm.aliasComp = [];
                ghmm.B = [];
                ghmm.smu = [];
                ghmm.ssigma = [];
            end
        end
        
        function ghmm = calculateCompWeightsFromStationary( ghmm )
            if ~isempty( ghmm.B)
                ghmm.compWeights =  ghmm.B * ghmm.stationary;
            else
                fprintf( 'calculateCompWeightsFromStationary fail \n' )
            end
        end
        
        function ghmm = setStateToComp( ghmm, sToC)
            ghmm.stateToComp = sToC;
            u = histc( ghmm.stateToComp,  unique( ghmm.stateToComp ) );
            ghmm.aliasComp = find(u > 1 );
            [B, ~ ] = ghmm.constructBC2alias( ghmm.aliasComp, [] );
            ghmm.B = B;
            ghmm.smu = ghmm.mu * ghmm.B;
            ghmm.ssigma = ghmm.sigma * ghmm.B;
        end
        
        function newHmm = matchHmmToHmm(hmm, hmmG)
            % match the states of the hmm to that of hmmG according to
            % least squares metric
            [~, ~,perm]  = bestPermMat( hmmG.A, hmm.A);
            newHmm = hmm.permuteStates( perm );
        end
        
        function newHmm = permuteStates(hmm, perm )
            newHmm = hmm;
            if ~isempty( newHmm.A )
                nA = newHmm.A(:,  perm);
                nA = nA(perm, :);
                newHmm.A = nA;
            end
            if ~isempty( newHmm.initial )
                nInit = newHmm.initial( perm);
                newHmm.initial = nInit;
            end
            if ~isempty( newHmm.stationary )
                nStat = newHmm.stationary( perm);
                newHmm.stationary = nStat;
            end
%             if ~isempty( newHmm.mu )
%                 nMu = newHmm.mu( perm);
%                 newHmm.mu = nMu;
%             end
%             if ~isempty( newHmm.sigma )
%                 nSigma = newHmm.sigma( perm);
%                 newHmm.sigma = nSigma;
%             end
        end
        
        %%% Genereting Data %%%%%%%%%%%
        function [states,observables] = generateSamples(ghmm,NumOfSamples)
            states = zeros(1,NumOfSamples);
            observables = zeros(1,NumOfSamples);
            states(1) = ghmm.sampleDiscrete(0);
            observables(1) = ghmm.sampleContinuous(states(1));
            for t = 2:NumOfSamples
                states(t) = ghmm.sampleDiscrete(states(t-1));
                observables(t) = ghmm.sampleContinuous(states(t));
            end
        end
        
        function ns = sampleDiscrete(ghmm,s)
            % sample next state from s
            if s ~= 0
                [~, ns] = histc(rand(1),ghmm.Acum(:,s));
            else
                [~, ns] = histc(rand(1),ghmm.initialCum);
            end
        end
        
        function o = sampleContinuous(ghmm,s)
            % sample output according to output distribution of state s
            comp = ghmm.stateToComp(s);
            o = normrnd(ghmm.mu( comp ),ghmm.sigma( comp ));
        end
        %%%%%%%%%%%%%%%%%%%%%%%%%%%
        
        function f = kernelCalculate(ghmm)
            % calculate the matrix of inner products between the output
            % distribution (analyticaly)
            f = zeros(ghmm.numComp);
            for i = 1:ghmm.numComp
                for j = 1:ghmm.numComp
                    f(i,j) =  1/(sqrt(2* pi* (ghmm.sigma(i)^2 + ghmm.sigma(j)^2)))...
                        *exp(-.5 *(ghmm.mu(i)-ghmm.mu(j))^2 /(ghmm.sigma(i)^2 + ghmm.sigma(j)^2));
                end
            end
        end
        
        function stat = stationaryCalculate(ghmm)
            % calculate the stationary distribution from transition matrix
            [vectors eigens] = eig(ghmm.A);
            e =max(max(abs(eigens)));
            [l m] = find(eigens == e);
            stat = vectors(:,l);
            stat  = stat/sum(stat);
        end
        
        function initial = initialDistCalculate( hmm, dataPrm)
            init_state_distrib = hmm.stationary;
            transmat = hmm.A';
            obslik = hmm.B' * dataPrm(: ,1); % An approximation!
            [~, ~, gamma] = fwdback(init_state_distrib, ...
                transmat, obslik);
            initial = gamma(:,1);
        end
        
        function [B,C] = constructBC( ghmm, sToC, stat )
            % lifting and projection matrices
            % B and C are as in the paper
            if isempty( sToC )
                sToC = ghmm.stateToComp;
                numC = ghmm.numComp;
            else
                numComp = length( unique( sToC ) );
            end
            if isempty( stat )
                stat = ghmm.stationary;
                n = ghmm.n;
            else
                n = length( stat );
            end
            B = zeros(  numC, n ) ;
            idx = sub2ind( size(B) , sToC, [1:n]);
            B( idx ) = 1               ;
            C = B';
            C = diag( stat./ (sum( diag( B * stat ) * B ) )' ) * C;
        end
        
        function [B,C] = constructBC2alias( ghmm, aliasComp, b ) % B and C are as in the paper
            stateToComp = ghmm.stateToComp;
            B = zeros( ghmm.numComp , ghmm.n );
            idx = sub2ind( size(B) , stateToComp , [1:ghmm.n]);
            B( idx ) = 1;
            C = [];
            if ~isempty( b )
                conditionals = ones( ghmm.n, 1 );
                conditionals( aliasComp ) = b;
                conditionals( aliasComp + 1 ) = 1 - b;
                newStat = diag( conditionals ) * B' * ghmm.compWeights;
                C = B';
                C = diag( newStat./ (sum( diag( B * newStat ) * B ) )' ) * C;
            end
        end
        
        function dataPrm = parametrizeData(ghmm,data)
            % apply the numComp different output distribution pdfs on the data
            dataPrm = repmat(data,ghmm.numComp,1);
            for i = 1 : ghmm.numComp
                dataPrm(i,:) = arrayfun(ghmm.pdfs{i},dataPrm(i,:)) ;%.* ghmm.stationary(i);
            end
        end
        
        function momSO = estimatePairMomentsSO(ghmm,dataPrm)
            % estimate the second order moments, averaging over all consequtive
            % pairs
            momSO = zeros( ghmm.numComp, ghmm.numComp) ;
            momSO(:,:) = ( (dataPrm(:,1:(end-1)) * dataPrm(:,2:end)')./(size(dataPrm,2)-1) )';
        end
        
        
        
        function momSO = estimateMomentsSO(ghmm,dataPrm)
            % estimate the second order moments with time lags 1,2,3
            momSO = zeros( ghmm.numComp, ghmm.numComp,  3) ;
            momSO(:,:,1) = ( (dataPrm(:,1:(end-1)) * dataPrm(:,2:end)')./(size(dataPrm,2)-1) )';
            momSO(:,:,2) = ( (dataPrm(:,1:(end-2)) * dataPrm(:,3:end)')./(size(dataPrm,2)-2) )';
            momSO(:,:,3) = ( (dataPrm(:,1:(end-3)) * dataPrm(:,4:end)')./(size(dataPrm,2)-3) )';
        end
        
        function momTO = estimateMomentsTO(ghmm,dataPrm)
            % estimate third order moments
            momTO = zeros( ghmm.numComp, ghmm.numComp, ghmm.numComp ) ;
            for i = 1 : ghmm.numComp
                momTO(:,:,i) =  ( (dataPrm(:,1:(end-2)) .* repmat( (dataPrm(i,2:(end-1))), ghmm.numComp,1 ) )  ...
                    * (dataPrm(:,3:end)') ./(size(dataPrm,2)-2) )';
            end
        end
        
        function loglik = calculateLoglik( hmm, data )
            % calculate the likelihhod of the data according to hmm
            O = 1;
            M = 1;
            if isempty( hmm.initial )
                cprior = hmm.stationary;
            else
                cprior = hmm.initial;
            end
            ctransmat = hmm.A';
            cmu = reshape( hmm.smu , [O hmm.n M]);
            cSigma =reshape( hmm.ssigma, [O O hmm.n M] );
            cMixmat = ones(hmm.n , M);
            loglik = mhmm_logprob(data, cprior, ctransmat, cmu, cSigma, cMixmat);
        end
        
        function path = viterbi( hmm, data )
            dataPrm = hmm.parametrizeData( data );
            prior = hmm.stationary;
            transmat = hmm.A';
            obslik = hmm.B' * dataPrm; % An approximation!
            path = viterbi_path(prior, transmat, obslik);
        end
        
        function newHmm = learnNonAliasA( hmmKnown,  data )
            %             global useDataFromFile % set true if data is given from file
            %             if useDataFromFile
            %                 load('data')
            %             else
            dataPrm = parametrizeData(hmmKnown,data);
            %            end
            momSO = hmmKnown.estimatePairMomentsSO(dataPrm);
            
            n = hmmKnown.numComp;
            mu = hmmKnown.mu;
            sigma = hmmKnown.sigma;
            compWeights = hmmKnown.compWeights;
            stateToComp = 1:n;
            A = [];
            newHmm = AliasGHMM.NonAliasGHMMe( n, [], mu, sigma, compWeights);
            % A1 preliminary %%%%%%%
            A = LearnTransitionQP( newHmm, momSO );  % QP as in ICML13
            newHmm.A = A;
            newHmm.stationary = newHmm. stationaryCalculate();
            newHmm.initial =  newHmm.initialDistCalculate(  dataPrm );
        end
        
        function result = LearnTransitionQP(ghmm,T)
            numC = ghmm.numComp;
            for i = 1 : numC
                for j = 1 : numC
                    for k = 1 : numC
                        for l = 1 : numC
                            cc(i,j,l,k) = ghmm.compWeights(j) * ghmm.kernel(k,j) * ghmm.kernel(l,i);
                        end
                    end
                end
            end
            for l =1 :numC
                for k =1:numC
                    temp(:,l,k)=reshape(cc(:,:,l,k),[numC^2,1]);
                end
            end
            c=reshape(temp,[numC^2,numC^2])';
            t = reshape(T,[numC^2,1]);
            H = c' * c;
            f = -t' * c;
            A =  [];
            b =  [];
            Aeq = zeros(2*numC, numC^2);
            for i = 1:numC
                Aeq(i,(numC*(i-1)+1):(numC*i)) = ones(1,numC);
                Aeq( numC+i  , i : numC : end ) = ghmm.compWeights;
            end
            beq = [ones(numC,1); ghmm.compWeights];
            lb = zeros(numC^2,1);
            ub = [];
            options = optimset('Display','on','Algorithm', 'interior-point-convex' ...
                ,'TolX',1e-10,'TolFun',1e-10,'MaxIter',100000);%,'Algorithm','interior-point-convex');
            [x, ~, exitFlag] = quadprog(H,f,A,b,Aeq,beq,lb,ub,[],options);
            result = reshape(x,[numC,numC]);
            if exitFlag ~= 1
                %fprintf('Optimization failed \n');
            end
        end
        
        
        function newHmm = learnAliasA( hmmKnown,  data )
            % estimate moments %%%%%%%%%%%%%%%%%%%%%
            %           global useDataFromFile
            %           if useDataFromFile
            %               load('data')
            %           else
            dataPrm = parametrizeData(hmmKnown,data);
            %           end
            
            momSO  = hmmKnown.estimateMomentsSO( dataPrm ) ;% second order moments
            momTO = hmmKnown.estimateMomentsTO( dataPrm ); % third order moments
            
            % striped moments %%%%%%%%%%%%%%%%%%%%%
            SmomSO = zeros( hmmKnown.numComp, hmmKnown.numComp, 3); % 3 - for three time legs
            SmomTO = zeros( hmmKnown.numComp, hmmKnown.numComp, hmmKnown.numComp);
            
            %%%%  DEBUG: WORKING WITH EXACT MOMENTS %%%%%%%
            %                         aliasComp = 1
            %                         b  = hmmKnown.stationary( 1 )/ sum( hmmKnown.stationary( 1:2 ) )
            %                         m = hmmKnown.numComp
            %                         n = hmmKnown.n
            %                         realB =  zeros( m , n )
            %                         idx = sub2ind( size( realB ) , hmmKnown.stateToComp , [1:n])
            %                         realB( idx ) = 1
            %                         conditionals = ones( n, 1 )
            %                         conditionals( aliasComp ) = b
            %                         conditionals( aliasComp + 1 ) = 1 - b
            %                         newStat = diag( conditionals ) * realB' * hmmKnown.compWeights
            %                         realC = realB'
            %                         realC = diag( newStat./ (sum( diag( realB * newStat ) * realB ) )' ) * realC
            %                         %realK =  realB' * dataHmm.kernel *realB
            %                         %realK = diag( dataHmm.kernel(k,:) )
            %                         clear inx be aliasComp conditionals newStat n m
            
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            
            for k = 1 : hmmKnown.numComp
                if k <= 3
                    SmomSO(:, :, k) = hmmKnown.stripMoments( momSO(:,:,k) );
                    
                    %%%% DEBUG: WORKING WITH EXACT MOMENTS %%%%%%%%%
                    %SmomSO(:, :, k) = realB * (hmmKnown.A ^ k ) * realC
                    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
                end
                SmomTO(:, :, k) = hmmKnown.stripMoments( momTO(:,:,k) );
                
                %%%% DEBUG: WORKING WITH EXACT MOMENTS %%%%%%%%%
                %SmomTO(:, :, k) = realB * hmmKnown.A *...
                %  diag( hmmKnown.kernel(k,:) * realB )   ...
                %    * hmmKnown.A * realC
                %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            end
            
            deltaM = SmomSO(:,:,2) - ( SmomSO(:,:,1) )^2;
            [U,S,V] = svd( deltaM );
            
            % detect aliasing %%%%%%%%%%%%%%%%%%%%%%
            ths = hmmKnown.getThreshold( size( dataPrm, 2 )); % need to change to c/sqrt(T) for suitable c
            aliasOrder = hmmKnown.detectAlias( S ,ths);
            
            % identify aliasing states %%%%%%%%%%%%%%%%%
            if aliasOrder > 0
                aliasComp = hmmKnown.identAliasComp( deltaM, SmomTO, SmomSO, aliasOrder );
            else
                aliasComp = [];
            end
            % reconstruct A %%%%%%%%%%%%%%%%%%%%%%%%
            % A = A1 + A2 + A3 +A4  as in the paper %%%%%%%%
            
            % construct new hmm with known parameters %%%%%
            m = hmmKnown.numComp;
            n = m + aliasOrder;
            mu = hmmKnown.mu;
            sigma = hmmKnown.sigma;
            compWeights = hmmKnown.compWeights;
            if ~isempty( aliasComp )
                stateToComp = [ 1 : aliasComp, aliasComp : m ];
            else
                stateToComp = 1:n;
            end
            A = [];
            newHmm = AliasGHMM( n, [] , stateToComp, mu, sigma, compWeights, [], [] );
            
            % A1 preliminary %%%%%%%
            BAC = LearnTransitionQP( newHmm, momSO(:, :, 1) );  % QP as in ICML13
            
            %%%% DEBUG: WORKING WITH EXACT MOMENTS %%%%%%%%%
            %BAC = SmomSO(:,:,1)
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            
            % Amax %%%%%%%%%%%%%%%%%%%%
            v =U(:,1);
            u = V(:,1);
            s =S(1);
            
            % Explicit solution %%%%%%%%%%%%%%%%%%
            %          g1max = min( ( .5*sign(v) + .5 - BAC(:, aliasComp ) )./ v )
            %          g2max = -max( ( -.5*sign(v) + .5 - BAC(:,aliasComp) ) ./ v )
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%
            
            % LP solution %%%%%%%%%%%%%%
            f = [-1, -1];
            Aconstrain = zeros( 4 * m, 2);
            Aconstrain( (1:m), 1 ) = - v;
            Aconstrain( m + (1:m), 1 ) = v;
            Aconstrain( 2*m + (1:m), 2 ) = v;
            Aconstrain( 3*m + (1:m), 2 ) =  - v;
            bconstrain = zeros( 4 * m, 1);
            bconstrain( 1:m) = BAC(:,1);
            bconstrain(m +  (1:m)) = 1 - BAC(:,1);
            bconstrain( 2*m +  (1:m)) = BAC(:,1);
            bconstrain( 3*m +  (1:m)) =  1 - BAC(:,1);
            lb = zeros( 2,1 );
            ub = [];
            Aeq = [];
            beq = [];
            x = linprog( f , Aconstrain , bconstrain , Aeq , beq , lb , ub, 'TolFun',1e-26 );
            g1max = x(1);
            g2max = x(2);
            g = g1max + g2max;
            b = g2max/g;
            Amax = newHmm.constructA( b , g , U, S, V, BAC, SmomSO, SmomTO, aliasComp);
            newHmm.Amax = Amax;
            if  0% min( min ( Amax ) >= 0 )
                A = Amax;
            else
                % Amin %%%%%%%%%%%%%%%%%%%%
                f = [1, 1];
                Aconstrain = zeros( 4 * m, 2);
                Aconstrain( (1:m), 2 ) = - BAC(1,:)';
                Aconstrain( m + (1:m), 1 ) =  - 1;
                Aconstrain( m + (1:m), 2 ) =  BAC(1,:)' - 1;
                Aconstrain( 2*m + (1:m), 1 ) = - BAC(1,:)';
                Aconstrain( 3*m + (1:m), 1 ) =  BAC(1,:)' - 1;
                Aconstrain( 3*m + (1:m), 2 ) =  - 1;
                bconstrain = zeros( 4 * m, 1);
                bconstrain( 1:m) = s * u;
                bconstrain(m +  (1:m)) = - s * u;
                bconstrain( 2*m +  (1:m)) = - s * u;
                bconstrain( 3*m +  (1:m)) =  s * u;
                lb = zeros( 2,1 );
                ub = [];
                Aeq = [];
                beq = [];
                x = linprog( f , Aconstrain , bconstrain , Aeq , beq , lb , ub );
                g1min = x(1);
                g2min = x(2);
                g = g1min + g2min;
                b = g2min/(g1min+g2min);
                Amin = newHmm.constructA( b , g , U, S, V, BAC, SmomSO, SmomTO, aliasComp);
                newHmm.Amin = Amin;
                if  0 %min( min ( Amin ) >= 0 )
                    A = Amin;
                else
                    % Afeas %%%%%%%%%%%%%%%%%%%%
                    x0 = [ g1min, g1max ];
                    A = [];
                    b = [];
                    Aeq = [];
                    beq = [];
                    
                    lb = [ min(g1min,g1max), min(g2min, g2max) ];
                    ub = [max(g1min,g1max), max(g2min, g2max) ];
                    fun = @(x) newHmm.feasObj( x, U, S , V , BAC, SmomSO, SmomTO, aliasComp );
                    [x,~,exitflag] = fmincon(fun,x0,A,b,Aeq,beq,lb,ub);
                    g1feas = x(1);
                    g2feas = x(2);
                    Afeas = newHmm.constructA(  x(2)/sum(x) , sum(x) , U , S , V , BAC, SmomSO, SmomTO, aliasComp);
                    if exitflag>0
                        A = Afeas;
                    else
                        A = Amax;
                    end
                end
            end
            A(A<0) = 0;
            A = mk_stochastic(A')';
            newHmm.A = A;
            newHmm.stationary = newHmm. stationaryCalculate();
            newHmm.initial =  newHmm.initialDistCalculate(  dataPrm );
            %%%% plot simplex %%%%
            %             hold off;
            %             clf;
            %             plot3( [1;0;0;1],[0;1;0;0],[0;0;1;0],'k.-' )
            %             view([135,25])
            %             daspect([1,1,1])
            %%%%%%%%%%%%%%%%%%%%%
            %             BA = B * dataHmm.A
            %             BAe = B * A
            %             BAC = SmomSO(:,:,1)
            %             BAC2 = B * A * C
            %             BAAC = SmomSO(:,:,2)
            %             BACBAC = BAC^2
            %             BAAAC = SmomSO(:,:,3)
            %             dAAA = BAAAC - BAAC*BAC - BAC*BAAC + BAC^3
            %             hold on;
            %             scatter3( BACBAC(1,:)'  , BACBAC(2,:)' , BACBAC(3,:)', 'fill')
            %             hold on
            %             scatter3( BAAC(1,:)'  , BAAC(2,:)' , BAAC(3,:)', 'fill', 'r')
            %              scatter3( dAAA(1,:)'  , dAAA(2,:)' , dAAA(3,:)', 'fill', 'g')
            %             scatter3( BAC(1,:)'  , BAC(2,:)' , BAC(3,:)', 'fill', 'g')
            %             scatter3( BAC2(1,:)'  , BAC2(2,:)' , BAC2(3,:)', 'fill', 'y')
            %             scatter3( BA(1,:)'  , BA(2,:)' , BA(3,:)', 'fill', 'm')
            %             scatter3( BAe(1,:)'  , BAe(2,:)' , BAe(3,:)',  'fill', 'dk')
            %             %scatter3( a1(1,:)'  , a1(2,:)' , a1(3,:)', 'fill', 'k')
            %             %scatter3( a2(1,:)'  , a2(2,:)' , a2(3,:)', 'fill', 'k')
            %             axis( [0 1 0 1 0 1] )
            %            [Amax; Amin; Afeas]
            %feasibleRegion(dataHmm.A,[g1min, g2min; g1max g2max; g1feas, g2feas])
            drawnow
        end
        
        function ths = getThreshold(hmm, T)
            ths = 1.8/(T^(1/3));
        end
        
        function aliasOrder = detectAlias( hmm, S, ths)
            SVtrun = S;
            SVtrun( SVtrun < ths ) = 0;
            aliasOrder = rank(SVtrun);
        end
        
        function aliasComp = identAliasComp( hmmKnown,deltaM, SmomTO, SmomSO, aliasOrder )
            for k = 1 : hmmKnown.numComp
                KsMat( :, :, k ) = SmomTO( : , : ,k) - SmomSO( : , : , 1) * diag( hmmKnown.kernel( k , : ) ) * SmomSO( : , : , 1);
                for c = 1 : hmmKnown.numComp
                    Sc(k, c) = norm( KsMat(:,:,k) -  hmmKnown.kernel( k , c ) * deltaM, 'fro' )^2;
                end
            end
            Sck = sum( Sc );
            [ ~ , aliasComp] = min( Sck );
        end
        
        
        function r = feasObj( ghmm,  x, U, S , V , BAC, SmomSO, SmomTO, aliasComp )
            A = constructA( ghmm,  x(2)/sum(x) , sum(x) , U , S , V , BAC, SmomSO, SmomTO, aliasComp);
            r = -min( min( A(1:2,1:2)  ) );
        end
        
        function A = constructA( ghmm,  b , g, U , S , V , BAC, SmomSO, SmomTO, aliasComp)
            v =U(:,1);
            u = V(:,1);
            s =S(1);
            deltaM = U * S * V';
            da = g * v;
            dal = u * s / g;
            %%%%% construct new  B C %%%%%%%%%%%
            [B, C] = ghmm.constructBC2alias( aliasComp , b  );
            
            A1 = C * BAC * B;
            vec = zeros( ghmm.n, 1 );
            vec( aliasComp ) = 1-b;
            vec( aliasComp + 1 ) = -b;
            A2 =   C * da * vec'; %[ 1-be , -be, 0, 0 ]
            vec( aliasComp ) = 1;
            vec( aliasComp + 1 ) = -1;
            A3 = vec * dal' * B;
            
            % A4 preliminary %%%%%%%%%%%%%%%%%%%%%%%%%%
            res = SmomSO(:,:,3) ...
                - SmomSO(:,:,2)*SmomSO(:,:,1)...
                - SmomSO(:,:,1)*SmomSO(:,:,2)...
                + SmomSO(:,:,1)^3;
            c = v' * res * u / s;
            
            % A %%%%%%%%%%%%%%%%%%%%%%%%%%%
            A = ( A1 + A2 + A3 + c * ( eye( ghmm.n ) - C * B ));
        end
        
        function SM = stripMoments( ghmm, M )
            Kinv = ghmm.kernel^-1;
            compWeightsInv = diag ( ghmm.compWeights ) ^-1;
            SM  =  Kinv * M * Kinv * compWeightsInv;
        end
        
        function NonAliasHMM = reduceToNoNAliasGHMM( ghmm )
            n = ghmm.numComp;
            statCond =  ghmm.stationary(  ghmm.stateToComp == ghmm.aliasComp  )
            b = statCond( 1 ) / sum( statCond );
            [B, C] = ghmm.constructBC2alias( ghmm.aliasComp , b  );
            A = B * ghmm.A * C;
            sToC = [1:n];
            initial = B * ghmm.initial;
            stationary = [];
            NonAliasHMM =  AliasGHMM( n , A , sToC , ghmm.mu, ghmm.sigma, ghmm.compWeights, initial, stationary);
        end
        function calculateSigmaR2(hmm)
            % assume aliased states 1,2
            if isempty(hmm.stationary)
                stationary =  hmm.stationaryCalculate();
            else
                stationary = hmm.stationary;
            end
            b = stationary(1) / sum(stationary(1:2));
            [B,C] =hmm.constructBC2alias( 1, b );
            A = hmm.A;
            R2 = B*A^2 * C - (B*A*C)^2;
            S = svd(R2);
        end
    end
    
    methods(Static)
        function ghmm = NonAliasGHMM( n, A , mu, sigma)
            sToC = 1:n;
            initial = [];
            stationary = [];
            compWeights = [];
            ghmm = AliasGHMM( n, A, sToC, mu,sigma, compWeights, initial, stationary);
        end
        
        function ghmm = NonAliasGHMMe( n, A , mu, sigma, compWeights)
            sToC = 1:n;
            initial = [];
            stationary = [];
            ghmm = AliasGHMM( n, A, sToC, mu,sigma, compWeights, initial, stationary);
        end
        
        
        function AliasHmm = generateAliasHMMExplicit( A, mu, sigma, stateToComp)
            n = size( A , 1);
            AliasHmm = AliasGHMM( n, A, stateToComp, mu, sigma, [], [], [] );
            AliasHmm = AliasHmm.calculateCompWeightsFromStationary();
        end
        
        
        function AliasHmm = generateAliasHMM( NumOfStates, numComp)
            %%% generates an aliased HMM for simulation example
            A = [...
                0.1 0.1 0.0 0.5
                0.7 0.1 0.3 0.0;
                0.0 0.8 0.1 0.25;
                0.2 0.0 0.6 0.25];
            
            n = NumOfStates;
            % components %%%%%%%%%%%%%%%%%%%%%
            if numComp == 3
                mu    = [0,3,6];
                sigma =1* [1,1,1];
                stateToComp = [1 1 2:numComp]; % assigning components to states
            end
            if numComp == 4
                mu    = [ -3, -2, 0, 3];
                sigma = .01 * [ 1, 1, 1, 1 ];
                stateToComp = [1 : numComp]; % assigning components to states
            end
            %%%%%%%%%%%
            
            % the hmm %%%%%%%%%%%%%%%%%%
            AliasHmm = AliasGHMM( n, A, stateToComp, mu, sigma, [], [], [] );
            AliasHmm = AliasHmm.calculateCompWeightsFromStationary();
        end
        
        function NonAliasHmm = generateNonAliasHMM( NumOfStatesG, NumCompG)
            %%% generates a non-aliased HMM for simulation example
            n = 4;
            A = [...
                0.7 0.0 0.2 0.5
                0.2 0.6 0.2 0.0;
                0.1 0.2 0.6 0.0;
                0.0 0.2 0.0 0.5];
            % output components %%%%%%%%%%%%%%%%%%%%%
            numComp = n;
            mu    = [ -4, 0, 2, 4];
            sigma = 1 * [ 2, 1, 6, 1 ];
            stateToComp = [1 : numComp]; % assigning components to states
            % the hmm %%%%%%%%%%%%%%%%%%
            NonAliasHmm = AliasGHMM.NonAliasGHMM( n, A, mu, sigma);
            NonAliasHmm.compWeights = NonAliasHmm.stationaryCalculate();
            %NonAliasHmm = NonAliasHmm.calculateCompWeightsFromStationary();
        end
    end
end

