#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "GIntervalsBigSet.h"
#include "GIntervalsBigSet1D.h"
#include "GIntervalsBigSet2D.h"
#include "rdbutils.h"

//------------------------------------- GIntervalsBigSet ----------------------------------------

const char *GIntervalsBigSet::META_FIELD_NAMES[NUM_META_FIELDS] = { "stats", "zeroline" };

SEXP GIntervalsBigSet::get_meta(const char *intervset, const IntervUtils &iu)
{
	string path = interv2path(iu.get_env(), intervset);
	path += "/.meta";
	SEXP meta = RSaneUnserialize(path.c_str());
	rprotect(meta);
	if (!isVector(meta) || length(meta) != 2)
		verror("Invalid format of intervals set %s", intervset);

	SEXP chromsizes = VECTOR_ELT(meta, 0);
	if (!isVector(chromsizes) || (length(chromsizes) != GIntervalsBigSet1D::NUM_STAT_COLS && length(chromsizes) != GIntervalsBigSet2D::NUM_STAT_COLS))
		verror("Invalid format of intervals set %s", intervset);

	return meta;
}

bool GIntervalsBigSet::isbig(const char *intervset, const IntervUtils &iu)
{
	string path = interv2path(iu.get_env(), intervset);
	bool interv_found = false;
	SEXP gintervs;

	rprotect(gintervs = findVar(install("GINTERVS"), iu.get_env()));
	for (int iinterv = 0; iinterv < length(gintervs); ++iinterv) {
		const char *interv = CHAR(STRING_ELT(gintervs, iinterv));
		if (!strcmp(intervset, interv)) {
			interv_found = true;
			break;
		}
	}

	if (!interv_found)
		return false;

	struct stat stat_res;

	return path.size() > rdb::INTERV_FILE_EXT.size() && !path.compare(path.size() - rdb::INTERV_FILE_EXT.size(), rdb::INTERV_FILE_EXT.size(), rdb::INTERV_FILE_EXT) &&
		!stat(path.c_str(), &stat_res) && S_ISDIR(stat_res.st_mode);
}

void GIntervalsBigSet::init(const char *intervset, const IntervUtils &iu)
{
	m_intervset = intervset;
	m_iu = (IntervUtils *)&iu;
}

void GIntervalsBigSet::save_meta(const char *intervset, SEXP stats, SEXP zeroline, const IntervUtils &iu)
{
	SEXP meta;
	SEXP colnames;

	rprotect(meta = allocVector(VECSXP, NUM_META_FIELDS));

	SET_VECTOR_ELT(meta, STATS_FIELD, stats);
	SET_VECTOR_ELT(meta, ZEROLINE_FIELD, zeroline);

	setAttrib(meta, R_NamesSymbol, (colnames = allocVector(STRSXP, NUM_META_FIELDS)));

	for (int i = 0; i < NUM_META_FIELDS; ++i) 
		SET_STRING_ELT(colnames, i, mkChar(META_FIELD_NAMES[i]));

	string filename = interv2path(iu.get_env(), intervset);
	filename += "/.meta";
	RSaneSerialize(meta, filename.c_str());
}
