#ifndef _GINTERVALSBIGSET2D_H_INCLUDED_
#define _GINTERVALSBIGSET2D_H_INCLUDED_

#include "GIntervals2D.h"
#include "GIntervalsBigSet.h"
#include "GIntervalsFetcher2D.h"
#include "rdbinterval.h"

//------------------------------------- GIntervalsBigSet2D --------------------------------------
// !!!!!!!!! IN CASE OF ERROR THIS CLASS THROWS TGLException  !!!!!!!!!!!!!!!!

class GIntervalsBigSet2D : public GIntervalsBigSet, public GIntervalsFetcher2D {
public:
	enum StatCols {
		CHROM1_COL, CHROM2_COL, CONTAINS_OVERLAPS_COL, SIZE_COL, SURFACE_COL, NUM_STAT_COLS
	};

	struct ChromStat {
		bool    contains_overlaps;
		size_t  size;
		double  surface;

		ChromStat() : contains_overlaps(false), size(0), surface(0.) {}
	};

	static const char *STAT_COL_NAMES[NUM_STAT_COLS];

	GIntervalsBigSet2D() : GIntervalsFetcher2D(BIGSET2D) {}
	GIntervalsBigSet2D(const char *intervset, SEXP meta, const IntervUtils &iu) : GIntervalsFetcher2D(BIGSET2D) { init(intervset, meta, iu); }
	virtual ~GIntervalsBigSet2D() {}

	void init(const char *intervset, SEXP meta, const IntervUtils &iu);

	int64_t get_num_intervals(int chromid1, int chromid2) const { return m_chroms2size[chroms2idx(chromid1, chromid2)]; }

	void load_chrom(int chromid1, int chromid2);

	const GIntervals2D &get_chrom_intervals() { return m_intervals; }

	static bool is2d(SEXP meta) { return length(VECTOR_ELT(meta, 0)) == NUM_STAT_COLS; }

	static pair<ChromPair, ChromStat> get_chrom_stat(GIntervalsFetcher2D *intervals, const IntervUtils &iu);

	static void begin_save(const char *intervset, const IntervUtils &iu, vector<ChromStat> &chromstats);

	// saves generic intervals with optional additional columns
	static void save_chrom(const char *intervset, GIntervalsFetcher2D *intervals, SEXP rintervals, const IntervUtils &iu, vector<ChromStat> &chromstats);

	// ends save for generic intervals with optional additional columns
	static void end_save(const char *intervset, SEXP zeroline, const IntervUtils &iu, const vector<ChromStat> &chromstats);

	// saves plain intervals
	static void save_chrom_plain_intervals(const char *intervset, GIntervals2D &intervals, const IntervUtils &iu, vector<ChromStat> &chromstats);

	// ends save for plain intervals
	static void end_save_plain_intervals(const char *intervset, const IntervUtils &iu, const vector<ChromStat> &chromstats);

	static int chroms2idx(int chromid1, int chromid2, int num_chroms) { return chromid1 * num_chroms + chromid2; }
	static int idx2chrom1(int idx, int num_chroms) { return idx / num_chroms; }
	static int idx2chrom2(int idx, int num_chroms) { return idx % num_chroms; }

	//-------------------------------- GIntervalsFetcher2D interface -----------------------------------

	virtual GIntervalsFetcher2D *create_masked_copy(const set<ChromPair> &chrompairs_mask) const;

	virtual void seal() {}

	virtual size_t size() const { return m_size; }

	virtual size_t size(int chromid1, int chromid2) const { return m_chroms2size[chroms2idx(chromid1, chromid2)]; }

	virtual int num_chrom_pairs() const;

	virtual double surface() const { return m_surface; } // complexity: O(1)
	virtual double surface(int chromid1, int chromid2) const { return m_surfaces[chroms2idx(chromid1, chromid2)]; } // complexity: O(1)

	virtual void begin_iter();

	virtual void begin_chrom_iter(int chromid1, int chromid2);

	virtual bool next();
	virtual bool next_in_chrom();

	virtual bool isend() const { return m_iinterval >= m_intervals.end(); }
	virtual bool isend_chrom() const { return m_iinterval >= m_intervals.end() || m_cur_chromid != m_iter_chromid; }

	virtual GIntervals2D::const_iterator get_chrom_begin() const { return m_intervals.begin(); }
	virtual GIntervals2D::const_iterator get_chrom_end() const { return m_intervals.end(); }

	virtual bool get_next_chroms(int *chromid1, int *chromid2);

	virtual size_t iter_index() const { return m_iter_index; }

	virtual size_t iter_chrom_index() const { return m_iter_chrom_index; }

	virtual const GInterval2D &cur_interval() const { return *m_iinterval; }

	virtual void sort(Compare_t = compare_for_sort);

	virtual void verify_no_overlaps(const GenomeChromKey &chromkey, const char *error_prefix = "") const;

private:
	GIntervals2D                 m_intervals;
	vector<int64_t>              m_chroms2size;
	vector<int64_t>              m_orig_chroms2size;
	vector<double>               m_surfaces;
	vector<bool>                 m_contains_overlaps;
	size_t                       m_size;
	double                       m_surface;
	GIntervals2D::const_iterator m_iinterval;
	int                          m_cur_chromid;
	int                          m_iter_chromid;
	size_t                       m_iter_index;
	size_t                       m_iter_chrom_index;
	Compare_t                    m_compare;
	bool                         m_do_sort;

	int chroms2idx(int chromid1, int chromid2) const { return chromid1 * m_iu->get_chromkey().get_num_chroms() + chromid2; }
	int idx2chrom1(int idx) const { return idx / m_iu->get_chromkey().get_num_chroms(); }
	int idx2chrom2(int idx) const { return idx % m_iu->get_chromkey().get_num_chroms(); }
};


//------------------------------------- IMPLEMENTATION ------------------------------------------

inline int GIntervalsBigSet2D::num_chrom_pairs() const
{
	int res = 0;
	for (vector<int64_t>::const_iterator ichroms2size = m_chroms2size.begin(); ichroms2size < m_chroms2size.end(); ++ichroms2size) {
		if (*ichroms2size) 
			++res;
	}
	return res;
}

inline bool GIntervalsBigSet2D::next()
{
	++m_iinterval;
	++m_iter_index;
	++m_iter_chrom_index;
	if (m_iinterval >= m_intervals.end()) {
		++m_cur_chromid;
		for (; m_cur_chromid < (int)m_chroms2size.size(); ++m_cur_chromid) {
			if (m_chroms2size[m_cur_chromid]) {
				int chromid1 = idx2chrom1(m_cur_chromid);
				int chromid2 = idx2chrom2(m_cur_chromid);

				load_chrom(chromid1, chromid2);
				m_iinterval = m_intervals.begin();
				break;
			}
		}
	}
	return isend();
}

inline bool GIntervalsBigSet2D::next_in_chrom()
{
	if (isend_chrom()) 
		return false;

	++m_iinterval;
	++m_iter_index;
	++m_iter_chrom_index;
	return !isend_chrom();
}

inline bool GIntervalsBigSet2D::get_next_chroms(int *chromid1, int *chromid2)
{
	if (*chromid2 < m_iu->get_chromkey().get_num_chroms() - 1)
		++*chromid2;
	else {
		++*chromid1;
		*chromid2 = 0;
	}
	return *chromid1 < m_iu->get_chromkey().get_num_chroms() && *chromid2 < m_iu->get_chromkey().get_num_chroms();
}

#endif

