% DISCLAIMER: This code was borrowed from IGL lab (see http://igl.ethz.ch/). 
% Please contact Alec Jacobson, jacobson@inf.ethz.ch, before using this 
% code outside of an informal setting, i.e. for comparisons in an 
% academic paper.

function [U,CSM] = arap(varargin)
  % ARAP Solve for the as-rigid-as-possible deformation accoring to
  % "As-rigid-as-possible Surface Modeling" by Sorkine and Alexa
  %
  % U = arap(V,F,b,bc) given a rest mesh (V,F) and list of constraint vertex
  % indices (b) and their new postions (bc) solve for pose mesh positions (U)
  %
  % U = arap(V,F,b,bc,'ParameterName','ParameterValue',...)
  %
  % Inputs:
  %   V  #V by dim list of rest domain positions
  %   F  #F by {3|4} list of {triangle|tetrahedra} indices into V
  %   b  #b list of indices of constraint (boundary) vertices
  %   bc  #b by dim list of constraint positions for b
  %   Optional:
  %     'V0' #V by dim list of initial guess positions
  %       dim by dim by #C list of linear transformations initial guesses,
  %       optional (default is to use identity transformations)
  %     'CovarianceScatterMatrix'
  %       followed by CSM a dim*n by dim*n sparse matrix, see Outputs
  %     'Groups'
  %       followed by #V list of group indices (1 to k) for each vertex, such 
  %       that vertex i is assigned to group G(i)
  %     'Tol'
  %       stopping critera parameter. If variables (linear transformation matrix
  %       entries) change by less than 'tol' the optimization terminates,
  %       default is 0.75 (weak tolerance)
  %     'MaxIter'
  %       max number of local-global iterations, default is 10
  %     'StretchRatio' 
  %       followed by a number between 0 (all rotation, default) or 1 (all
  %       stretch)
  % Outputs:
  %   U  #V by dim list of new positions
  %   CSM dim*n by dim*n sparse matrix containing special laplacians along the
  %     diagonal so that when multiplied by repmat(U,dim,1) gives covariance 
  %     matrix elements, can be used to speed up next time this function is
  %     called, see function definitions
  %
  % See also: takeo_arap
  %

% DISCLAIMER: This code was borrowed from IGL lab (see http://igl.ethz.ch/). 
% Please contact Alec Jacobson, jacobson@inf.ethz.ch, before using this 
% code outside of an informal setting, i.e. for comparisons in an 
% academic paper.


  % parse input
  V = varargin{1};
  F = varargin{2};
  b = varargin{3};
  bc = varargin{4};
  G = [];

  % number of vertices
  n = size(V,1);
  assert(max(b) <= n);
  assert(min(b) >= 1);
  % dimension
  dim = size(V,2);
  assert(dim == size(bc,2));

  % default is normal arap no stretching term
  stretch_ratio = 0;

  ii = 5;
  while(ii <= nargin)
    if strcmp(varargin{ii},'V0');
      ii = ii + 1;
      assert(ii<=nargin);
      U = varargin{ii};
    elseif strcmp(varargin{ii},'CovarianceScatterMatrix');
      ii = ii + 1;
      assert(ii<=nargin);
      CSM = varargin{ii};
    elseif strcmp(varargin{ii},'Groups');
      ii = ii + 1;
      assert(ii<=nargin);
      G = varargin{ii};
      if isempty(G)
        k = n;
      else
        k = max(G);
      end
    elseif strcmp(varargin{ii},'Tol');
      ii = ii + 1;
      assert(ii<=nargin);
      tol = varargin{ii};
    elseif strcmp(varargin{ii},'MaxIter');
      ii = ii + 1;
      assert(ii<=nargin);
      max_iterations = varargin{ii};
    elseif strcmp(varargin{ii},'StretchRatio')
      ii = ii + 1;
      assert(ii<=nargin);
      stretch_ratio = varargin{ii};
    end
    ii = ii + 1;
  end

  if(~exist('U','var') || isempty(U))
    if(dim == 2) 
      U = laplacian_mesh_editing(V,F,b,bc);
    else
      U = V;
    end
  end
  assert(n == size(U,1));
  assert(dim == size(U,2));

  if(~exist('L','var'))
    if(size(F,2) == 3)
      L = cotmatrix(V,F);
    elseif(size(F,2) == 4)
      L = cotmatrix3(V,F);
    else
      error('Invalid face list');
    end
  end

  if(~exist('interior','var'))
    indices = 1:n;
    interior = indices(~ismember(indices,b));
  end
  all = [interior b];

  % cholesky factorization
  %cholL = chol(-L(interior,interior),'lower');
  [luL,luU,luP,luQ,luR] = lu(-L(interior,interior));

  %R = repmat(eye(dim,dim),[1 1 n]);
  if(~exist('max_iterations','var'))
    max_iterations = 100;
  end

  h = avgedge(V,F);
  if(~exist('tol','var'))
    tol = 0.001;
  end


  % build covariance scatter matrix used to build covariance matrices we'll
  % later fit rotations to
  if(~exist('CSM','var'))
    CSM = covariance_scatter_matrix(V,F);
    % if there are groups then condense scatter matrix to only build
    % covariance matrices for each group
    if ~isempty(G)
      G_sum = group_sum_matrix(G,k);
      %CSM = [G_sum sparse(k,n); sparse(k,n) G_sum] * CSM;
      CSM = repdiag(G_sum,dim) * CSM;
    end
  end

  % precompute rhs premultiplier
  [~,K] = arap_rhs(V,F,[]);

  % initialize rotations with identies (not necessary)
  R = repmat(eye(dim,dim),[1 1 n]);

  iteration = 0;
  U_prev = V;
  while( iteration < max_iterations && (iteration == 0 || max(abs(U(:)-U_prev(:)))>tol*h))
    U_prev = U;

    % energy after last global step
    U(b,:) = bc;
    %E = arap_energy(V,F,U,R);
    %[1 E]

    % compute covariance matrix elements
    S = CSM*repmat(U,dim,1);
    % dim by dim by n list of covariance matrices
    S = permute(reshape(S,[size(CSM,1)/dim dim dim]),[2 3 1]);
    % fit rotations to each deformed vertex
    R = fit_rotations(S);

    % energy after last local step
    U(b,:) = bc;
    %E = arap_energy(V,F,U,R);
    %[2 E]

    if(stretch_ratio > 0)
      St = fit_stretch(S);
    end


    % This is still the SLOW way of building the right hand side, the entire
    % step of building the right hand side should collapse into a
    % #handles*dim*dim+1 by #groups matrix times the rotations at for group
    
    % distribute group rotations to vertices in each group
    if ~isempty(G)
      R = R(:,:,G);
      if(stretch_ratio > 0)
        St = St(:,:,G);
      end
    end

    U(b,:) = bc;

    %B = arap_rhs(V,F,R);
    B = K * reshape(permute(R,[3 1 2]),n*dim*dim,1);
    if(stretch_ratio > 0)
      B = (1-stretch_ratio)*B + ...
        stretch_ratio* K * reshape(permute(St,[3 1 2]),n*dim*dim,1);
        error
    end
    B = reshape(B,[size(B,1)/dim dim]);

    %U(interior,:) = -L(interior,interior) \ (B(interior,:) + L(interior,b)*bc);
    %U(interior,:) = -L(interior,all)*L(all,interior) \ (L(interior,all)*B(all,:) + L(interior,all)*L(all,b)*bc);
    U(interior,:) = luQ*(luU\(luL\(luP*(luR\(B(interior,:)+L(interior,b)*bc)))));
    %U(interior,:)=cholL\((B(interior,:)+L(interior,b)*bc)'/cholL)';
    iteration = iteration + 1;
  end
end
