% DISCLAIMER: This code was borrowed from IGL lab (see http://igl.ethz.ch/). 
% Please contact Alec Jacobson, jacobson@inf.ethz.ch, before using this 
% code outside of an informal setting, i.e. for comparisons in an 
% academic paper.

function [b,K] = arap_rhs(V,F,R)
  % ARAP_RHS build right-hand side of global poisson solve for Sorkine and
  % Alexa's ARAP (see equation 8 and 9):
  %   ∑ wij * 0.5 * (Ri + Rj) * (Vi - Vj)
  % j∈N(i)
  % 
  % b = arap_rhs(V,F,R)
  % [b,K] = arap_rhs(V,F,R)
  %
  % Inputs:
  %   V  #V by dim list of initial domain positions
  %   F  #F by 3 list of triangle indices into V
  %   R  dim by dim by #F list of rotations, if R is empty then b is set to []
  %     and only K is computed
  % Output:
  %   b  #V by dim right hand side
  %   K  #V*dim by #V*dim*dim matrix such that: 
  %     b = K * reshape(permute(R,[3 1 2]),size(V,1)*size(V,2)*size(V,2),1);
  %   

  %
  % Bi =   ∑ wij * 0.5 * (Ri + Rj) * (Vi - Vj)
  %      j∈N(i)
  %
  % where Bi, Vi, and Vj are dim-length vectors, and Ri and Rj are
  % dim by dim rotation matrices
  %
  % Bi' =   ∑ wij * 0.5 * (Vi'-Vj') * (Ri' + Rj')
  %       j∈N(i)
  %
  % Bi(x) =   ∑ wij * 0.5 * ∑ (Vi(y)-Vj(y)) * (Ri(x,y) + Rj(x,y))
  %          j∈N(i)        y∈dim
  %
  % Where Bi(x) is getting the xth coordinate of Bi and Ri(x,y) is getting the
  % entry of Ri at row x and col y
  % 

  % DISCLAIMER: This code was borrowed from IGL lab (see http://igl.ethz.ch/). 
% Please contact Alec Jacobson, jacobson@inf.ethz.ch, before using this 
% code outside of an informal setting, i.e. for comparisons in an 
% academic paper.

  
  dim = size(V,2);

  
  KX = blck(V,F,1);
  KY = blck(V,F,2);
  Z = sparse(size(V,1),size(V,1));
  if dim == 2
    K = [ ...
      KX Z KY Z; ...
      Z KX Z KY];
  elseif dim == 3
    KZ = blck(V,F,3);
    K = [ ...
      KX Z Z  KY Z  Z  KZ Z  Z ; ...
      Z KX Z  Z  KY Z  Z  KZ Z ; ...
      Z Z  KX Z  Z  KY Z  Z  KZ ];
  end

  if(~isempty(R))
    assert(dim == size(R,1));
    assert(dim == size(R,2));
    % number of rotations
    nr = size(R,3);
    % collect rotations into a single column
    Rcol = reshape(permute(R,[3 1 2]),nr*dim*dim,1);
    b = K * Rcol;
    b = reshape(b,[size(V,1) 2]);
  else
    b = [];
  end

  function K = blck(V,F,d)
    % Computes a matrix K such that K * R computes
    %  ∑ wij * 0.5 * (V(i,d)-V(i,d)) * (Ri + Rj)
    % j∈N(i)
    % 
    % Inputs:
    %   V  #V by dim list of coordinates
    %   F  #F by 3 list of triangle indices into V
    %   d  index into columns of V
    % Output:
    %   K  #V by #V matrix
    %

    E = edges(F);
    % Build upper part of adjacency matrix where instead of a 1 for edge from i
    % to j we have the difference of position in dimension d
    A = sparse(E(:,1),E(:,2),V(E(:,1),d)-V(E(:,2),d),size(V,1),size(V,1));
    % anti-symmetric, or considers direction of edges
    A = A-A';
    % Multiply with cotangent weights (don't worry about diagonal begin wrong
    % since in A it's all zeros
    if(size(F,2) == 3)
      L = cotmatrix(V,F);
    elseif(size(F,2) == 4)
      L = cotmatrix3(V,F);
    else
      error('Invalid face list');
    end
    K = L.*A;
    % correct the diagonal (notice that the sign is positive
    K = K + diag(sum(K,2));
    K = 0.5*K;

    %if(size(F,2) == 3)
    %  i1 = F(:,1);
    %  i2 = F(:,2);
    %  i3 = F(:,3);


    %  % #F x 3 matrices of triangle edge vectors, named after opposite vertices
    %  v1 = V(i3,:) - V(i2,:);  v2 = V(i1,:) - V(i3,:); v3 = V(i2,:) - V(i1,:);
    %  % computing areas 
    %  if size(V,2) == 2
    %      % 2d vertex data
    %      dblA = v1(:,1).*v2(:,2)-v1(:,2).*v2(:,1);
    %  elseif size(V,2) == 3
    %      %n  = cross(v1,v2,2);  dblA  = multinorm(n,2);
    %      % area of parallelogram is twice area of triangle
    %      % area of parallelogram is || v1 x v2 || 
    %      n  = cross(v1,v2,2); 
    %      % THIS DOES MATRIX NORM!!! don't use it!!
    %      % dblA  = norm(n,2);
    %      % This does correct l2 norm of rows
    %      dblA = (sqrt(sum((n').^2)))';
    %  else 
    %      error('unsupported vertex dimension %d', size(V,2))
    %  end
    %  % cotangents and diagonal entries for element matrices
    %  % dot is 4 times slower than sum(.*)
    %  cot12 = sum(v1.*v2,2)./dblA/2; 
    %  cot23 = sum(v2.*v3,2)./dblA/2;
    %  cot31 = sum(v3.*v1,2)./dblA/2;

    %  % off diagonal entry parts for each triangle edge
    %  K12 = cot12.*(V(i1,d) - V(i2,d));
    %  K23 = cot23.*(V(i2,d) - V(i3,d));
    %  K31 = cot31.*(V(i3,d) - V(i1,d));

    %  % indices of nonzero elements in the matrix for sparse() constructor
    %  i = [i1 i3 i2];
    %  j = [i2 i1 i3];

    %  % values corresponding to pairs form (i,j)
    %  v = [K12 K31 K23];
    %  % for repeated indices (i,j) sparse automatically sums up elements, as we
    %  % want
    %  % number of vertices
    %  n = size(V,1);
    %  K = sparse(i,j,v,n,n);
    %  % anti-symmetry across edges
    %  K = K - K';
    %  K = K + diag(sum(K,2));
    %  K = -0.5*K;
    %else if(size(F,2) == 4))
    %  % tets
    %else
    %  error('Invalid face list');
    %end
  end

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% THIS IS AN OLD IMPLEMENTATION
%% It constructed b as b = Matrix(V,F,R) * V rather than Matrix(V,F) * R
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% function b = arap_rhs(V,F,R,L)
%   % ARAP_RHS build right-hand side of global poisson solve for Sorkine and
%   % Alexa's ARAP (see equation 8 and 9):
%   %   ∑ wij * 0.5 * (Ri + Rj) * (Vi - Vj)
%   % j∈N(i)
%   % 
%   % b = arap_rhs(V,F,R,L)
%   %
%   % Inputs:
%   %   V  #V by dim list of initial domain positions
%   %   F  #F by 3 list of triangle indices into V
%   %   R  dim by dim by #F list of rotations
%   %   Optional:
%   %     L  #V by #V sparse cotangent matrix
%   % Output:
%   %   b  #V by dim right hand side
%   %   
% 
%   if(~exist('L','var'))
%     L = cotmatrix(V,F);
%   end
% 
%   % In 2D:
%   % | LXX LXY | Vx = bx
%   % | LYX LYY | Vy   by
%   %
%   % Where LXXij = -wij*0.5*(R(1,1,i) + R(1,1,j))
%   % and LXXii = ∑ -LXXij
%   %           j∈N(i)
%   %
%   assert(size(V,2) == 2,'Only 2D is supported');
% 
%   LXX = laplacian(V,F,squeeze(R(1,1,:)));
%   LXY = laplacian(V,F,squeeze(R(1,2,:)));
%   % should be that LXY = -LXY
%   LYX = laplacian(V,F,squeeze(R(2,1,:)));
%   % should be that LYY = LXX
%   LYY = laplacian(V,F,squeeze(R(2,2,:)));
% 
%   % weight with cotangent matrix weights and divide by 2
%   % don't change signs!
%   LL = [LXX LXY; LYX LYY]*0.5;
%  
%   % multiple against positions
%   b = LL*[V(:,1); V(:,2)];
%   % reshape stacked vector into rows of column vectors
%   b = reshape(b,size(V));
% 
%   %test_b = zeros(size(V));
%   %for ii = 1:size(V,1)
%   %  for jj = 1:size(V,1)
%   %    test_b(ii,:) = test_b(ii,:) + ...
%   %      L(ii,jj) * 0.5 * ((R(:,:,ii) + R(:,:,jj)) * ((V(ii,:) - V(jj,:))'))';
%   %  end
%   %end
% 
%   %dim = size(V,2);
%   %test_LL = zeros([dim*size(V,1) dim*size(V,2)]);
%   %for ii = 1:size(V,1)
%   %  for jj = 1:size(V,1)
%   %    if(ii == jj)
%   %      for kk = 1:size(V,1)
%   %        if(kk ~= jj)
%   %          test_LL(((ii-1)*dim+1):(ii*dim),((ii-1)*dim+1):(ii*dim)) = ...
%   %            test_LL(((ii-1)*dim+1):(ii*dim),((ii-1)*dim+1):(ii*dim)) + ...
%   %            repmat(L(ii,kk),[dim dim]) .* (R(:,:,ii) + R(:,:,kk));
%   %        end
%   %      end
%   %    else
%   %      test_LL(((ii-1)*dim+1):(ii*dim),((jj-1)*dim+1):(jj*dim)) = ...
%   %        -repmat(L(ii,jj),[dim dim]) .* (R(:,:,ii) + R(:,:,jj));
%   %    end
%   %  end
%   %end
%   %test_LL = test_LL/2;
%   %%full(test_LL(1:2:(size(V,1)*dim),1:2:(size(V,1)*dim)))
%   %[b test_b reshape(test_LL*(reshape(V',1,numel(V))'),size(V,2),size(V,1))']
% 
% end
