% DISCLAIMER: This code was borrowed from IGL lab (see http://igl.ethz.ch/). 
% Please contact Alec Jacobson, jacobson@inf.ethz.ch, before using this 
% code outside of an informal setting, i.e. for comparisons in an 
% academic paper.

function [CSM] = covariance_scatter_matrix(varargin)
  % COVARIANCE_SCATTER_MATRIX
  %
  % CSM = covariance_scatter_matrix(V,F)
  % CSM = covariance_scatter_matrix(V,F,L,E)
  % 
  % Inputs:
  %   V  #V by dim list of initial domain positions
  %   F  #F by #simplex size list of triangle indices into V
  %   Optional:
  %     L  #V by #V sparse cotangent matrix
  %     E  #E by 2 edge list
  % Outputs:
  %   CSM dim*n by dim*n sparse matrix containing special laplacians along the
  %     diagonal so that when multiplied by repmat(U,dim,1) gives covariance 
  %     matrix elements, can be used to speed up covariance matrix computation
  % Example:
  %  % compute covariance matrices of a mesh
  %  n = size(V,1);
  %  dim = size(V,2);
  %  CSM = covariance_scatter_matrix(V,F)
  %  S = CSM*repmat(U,dim,1);
  %  % dim by dim by n list of covariance matrices
  %  S = permute(reshape(S,[n dim dim]),[2 3 1]);
  %
% DISCLAIMER: This code was borrowed from IGL lab (see http://igl.ethz.ch/). 
% Please contact Alec Jacobson, jacobson@inf.ethz.ch, before using this 
% code outside of an informal setting, i.e. for comparisons in an 
% academic paper.


  V = varargin{1};
  F = varargin{2};
  if(nargin > 2)
    L = varargin{3};
  else
    if(size(F,2) == 3)
      L = cotmatrix(V,F);
    elseif(size(F,2) == 4)
      L = cotmatrix3(V,F);
      %% 'wachspress'
      %L = L-diag(diag(L));
      %E = edges(F);
      %L = L.* ...
      %  sparse( ...
      %    [E(:,1);E(:,2)], ...
      %    [E(:,2);E(:,1)], ...
      %    repmat(sum((V(E(:,1),:)-V(E(:,2),:)).^2,2),2,1));
      %L = L - diag(sum(L,2));
      %L = adjacency_matrix(F);
      %L = L - diag(sum(L,2));
    else
      error('Invalid face list');
    end
  end

  % clamp laplacian (replace negative weights with 0s)
  L = L-diag(diag(L));
  % interpolate between fully clamped and unclamped
  t = 1;
  L = t.*(L>0).*L + (1-t).*L;
  L = L - diag(sum(L,2));

  if(nargin > 3)
    E = varargin{4};
  else
    E = edges(F);
  end
  % construct matrix such that multiplied against Us gives covariance
  % matrix elements
  dim = size(V,2);
  n = size(V,1);
  ne = size(E,1);
  I = zeros(2*ne*dim,1);
  J = zeros(2*ne*dim,1);
  v = zeros(2*ne*dim,1);
  for ii = 1:dim
    I((1+(ii-1)*2*ne):(ii*2*ne)) = (ii-1)*n + [E(:,1);E(:,2)];
    J((1+(ii-1)*2*ne):(ii*2*ne)) = (ii-1)*n + [E(:,2);E(:,1)];
    v((1+(ii-1)*2*ne):(ii*2*ne)) = [V(E(:,1),ii)-V(E(:,2),ii);V(E(:,2),ii)-V(E(:,1),ii)];
  end
  CSM = -repdiag(L,dim).*sparse(I,J,v,n*dim,n*dim);
  CSM = CSM + diag(sum(CSM));

  %else
  %  V = varargin{1};
  %  F = varargin{2};
  %  U = varargin{3};
  %  if(nargin > 3)
  %    L = varargin{4};
  %  end
  %  if(nargin > 4)
  %    A = varargin{5};
  %  end
  %  
  %  if(~exist('L','var'))
  %    L = cotmatrix(V,F);
  %  end
  %  if(~exist('A','var'))
  %    A = adjacency_matrix(F);
  %  end

  %  % number of vertices
  %  n = size(V,1);
  %  assert(n == size(U,1));
  %  % dimension
  %  dim = size(V,2);
  %  assert(dim == size(U,2));

  %  % covariance matrices
  %  %S = zeros([dim dim n]);
  %  S = cell(n,1);
  %  % loop over vertices
  %  for ii = 1:n
  %    % covariance matrix
  %    S{ii} = zeros([dim dim]);
  %    % neighbor indices
  %    neighbors = find(A(ii,:));
  %    for j = neighbors
  %      % rest edge vectors to neighbor j
  %      ev = V(j,:)-V(ii,:);
  %      % pose edge vectors to neighbor j
  %      eu = U(j,:)-U(ii,:);
  %      S{ii} = S{ii} + L(ii,j) .* ev' * eu;
  %    end
  %  end
  %  S = reshape(cell2mat(S'),[dim dim n]);
  %end
end
