% DISCLAIMER: This code was borrowed from IGL lab (see http://igl.ethz.ch/). 
% Please contact Alec Jacobson, jacobson@inf.ethz.ch, before using this 
% code outside of an informal setting, i.e. for comparisons in an 
% academic paper.

function [R] = fit_stretch(varargin)
  % FIT_stretch Given an input mesh and new positions find stretch for
  % every vertex that best maps its one ring to the new one ring
  % 
  % R = fit_stretch(V,F,U,'ParamName',ParamValue)
  % R = fit_stretch(S,'ParamName',ParamValue)
  %
  % Inputs:
  %   V  #V by dim list of initial domain positions
  %   F  #F by 3 list of triangle indices into V
  %   U  #V by dim list of new positions
  %     or
  %   S  dim by dim by #stretch list of covariance matrices to fit stretch
  %     to
  %   Optional parameters
  %     'AllowFlips'  optionally followed by true or false, find best fitting
  %       rotation OR reflection
  % Outputs:
  %   R  dim by dim by #F list of stretch
  %
% DISCLAIMER: This code was borrowed from IGL lab (see http://igl.ethz.ch/). 
% Please contact Alec Jacobson, jacobson@inf.ethz.ch, before using this 
% code outside of an informal setting, i.e. for comparisons in an 
% academic paper.


  if(nargin == 1)
    S = varargin{1};
    dim = size(S,1);
    assert(dim == size(S,2));
    ii = 2;
    nr = size(S,3);
  else
    assert(nargin == 3);
    V = varargin{1};
    F = varargin{2};
    U = varargin{3};
    ii = 4;

    % number of vertices
    n = size(U,1);
    % dimension
    dim = size(U,2);

    if ~exist('CSM','var')
      assert(n == size(V,1));
      assert(dim == size(V,2));
      CSM = covariance_scatter_matrix(V,F);
    end
    % number of stretch
    nr = size(CSM,1)/dim;
    % compute covariance matrix elements
    S = CSM*repmat(U,dim,1);
    % dim by dim by n list of covariance matrices
    S = permute(reshape(S,[nr dim dim]),[2 3 1]);
  end

  allow_flips = false;
  while(ii<=nargin)
    switch varargin{ii}
    case 'AllowFlips'
      if( (ii+1)<=nargin && ~ischar(varargin{ii+1}))
        ii = ii + 1;
        allow_flips = varargin{ii};
      else
        allow_flips = true;
      end
    otherwise
      error('Invalid parameter');
    end
    ii = ii+1
  end

  %R = cellfun(@fit_rotation,S,'UniformOutput',false);
  %R = reshape(cell2mat(R'),[dim dim nr]);
  % For loop is faster
  R = zeros([dim dim nr]);
  for ii = 1:nr
    % svd 
    [su,ss,sv]=svd(S(:,:,ii));
    %Ri = sv*su';
    ss = ss./min(diag(ss));
    min_diff = 2;
    ss = smooth((ss - eye(dim))/min_diff -1)*min_diff + eye(dim);
    max_ss = 10;
    ss = min(ss,max_ss);
    Ri = sv*su'*ss;
    % if reflection then flip last column
    if(~allow_flips && det(Ri) < 0 )
      su(:,end) = -su(:,end);
      Ri = sv*su';
    end
    % should definitely be rotation now
    %assert( det(Ri) >= 0 );
    R(:,:,ii) = Ri;
  end

  function D = det3(M)
    % DET3 compute the determinant of a 3x3 matrix
    % Input:
    %   M  3 by 3 matrix
    % Output:
    %   D  determinant
    %
    % http://en.wikipedia.org/wiki/Determinant#3-by-3_matrices
    D = ...
      M(1,1) * M(2,2) * M(3,3) + ...
      M(1,2) * M(2,3) * M(3,1) + ...
      M(1,3) * M(2,1) * M(3,2) - ...
      M(1,3) * M(2,2) * M(3,1) - ...
      M(1,2) * M(2,1) * M(3,3) - ...
      M(1,1) * M(2,3) * M(3,2);
  end

  function s = smooth(x)
    s = (0).*(x<0) + (x.^2/2).*(0 <= x & x < 1) + (x-1/2).*(1 <= x);
  end

end

