/**
 * MemoryCard.java
 **/
package il.ac.wis.cs.s2a.examples.memoryGame.classes;

import javax.swing.JComponent;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.Timer;
import java.util.TimerTask;

/**
 * MemoryCard represents a card in the Memory game.
 **/
public class MemoryCard extends JComponent implements MouseListener
{
  
	private static final long serialVersionUID = 1L;

	public void test(){};
	/** 
   * Description of this card (e.g., diamond, square, etc.).
   **/
  private String description; 
  
  /**
   * Color of this card.
   **/
  private Color color;

  /**
   * Show card face if true.
   **/
  public boolean faceUp;

  /**
   * Constructor. Creates a new MemoryCard instance with
   * the passed description and color, initially face down.
   **/
  public MemoryCard( String description, Color color )
  {
    this.description = description;
    this.color = color;

    // set card face down
    faceUp = false;
    
    addMouseListener(this);
  }

  public void flipUp() {
  	faceUp = true;
  }
  
  public void rePaint() {
  	repaint (0);
  }
  
  public void reDraw()
  {
	  rePaint();
  }
  /**
   * Invoked when the mouse button has been clicked 
   **/
  public void mouseClicked( MouseEvent e ) {}
  
  /**
   * Invoked when the mouse enters a component. 
   **/
  public void mouseEntered( MouseEvent e ) {}
  
  /**
   * Invoked when the mouse exits a component. 
   **/
  public void mouseExited( MouseEvent e ) {} 
  
  /**
   * Invoked when a mouse button has been pressed on a component. 
   **/
  public void mousePressed( MouseEvent e ) {} 
     
  /**
   * Invoked when a mouse button has been released on a component. 
   **/
  public void mouseReleased( MouseEvent e ) {} 

  /** 
   * Get description for this card.
   * @return String describing the card.
   **/
  private String getDescription()
  {
    return description;
  }

  /**
   * Get the color of this card.
   * @return card's Color.
   **/
  private Color getColor()
  {
    return color;
  }

  /**
   * Test if card is face up.
   * @return true if card is face up.
   **/
  public boolean isFaceUp()
  {
    return faceUp;
  }

  /**
   * Check if testCard matches this card in
   * description and color.
   * @return true if both description and color match;
   * false, otherwise.
   **/
  public boolean match( MemoryCard testCard )
  {
    return testCard.getDescription().equals( description ) 
      && testCard.getColor().equals( color );
  }

  /**
   * Special method as specified in JComponent (parent)
   * for drawing on this button.
   * We *override* the method here to customize the display.
   **/
  public void paint( Graphics g )
  {
    // if card is face up
    if ( faceUp )
    {
      // call utility method to paint background of card
      // with color
      // pass along Graphics object to paint on
      paintBackground( g, color );

      // call utility method to paint description onto card
      paintDescription( g );
    }
    // otherwise, card is face down
    else
    {
      // simply paint the back of a card
      paintBackground( g, Color.LIGHT_GRAY );
    }
  }

  /**
   * Utility method to paint the background of the card
   * with the specified color.
   **/
  private void paintBackground( Graphics g, Color c )
  {
    // set the color of the "brush"
    g.setColor( c );

    // fill a rectangle (using the card's color)
    // arguments are x, y, width, height
    // note: we use methods getWidth() and getHeight(),
    // which we inherited from JComponent
    g.fillRoundRect( 0 + 2, 0 + 2, getWidth() - 1 - 2, getHeight() - 1 - 2, 20, 20 );

    // add a rounded rectangular border in black
    g.setColor( Color.black );
    // arguments are x, y, width, height, arcWidth, arcHeight
    g.drawRoundRect( 0 + 2, 0 + 2, getWidth() - 1 - 2, getHeight() - 1 - 2, 20, 20 ); 
  }

  /**
   * Utility method to paint the description onto the card.
   **/
  private void paintDescription( Graphics g )
  {
    // set "brush" color to white
    g.setColor( Color.white );

    // set font
    g.setFont( new Font( "Arial", Font.BOLD, 24 ) );

    // draw text
    g.drawString( description, 10, getHeight() / 2 + 5 );
  }
  
  public void flipDown()
  {
    Timer timer = new Timer();
    timer.schedule(
        new TimerTask() 
        { 
          public void run()
          {
            faceUp = false;
            
            // update display
            repaint(0);
          }
        },
        800 );
  }
}
