
/*
 ** This file is the most important ingredient of the entire
 * package. It contains the module for evaluation of the OWF
 * once the set system and predicate have been generated. Any 
 * piece of code requiring to use the OWF (for eq. the PRF) must
 * make use of this module and only this module. 
 *
 * The file should not be moved out of the directory it is 
 * currently in because it has dependencies on other files in
 * the package. (Though this is important for every file in 
 * the package, we stress this here because the user would 
 * intend making use of this module in his programs.)
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "MathFunctions.hpp"
#include "ArrayFunctions.hpp"

#define genfile "data/setsystem.dat"  // File containing the default setsystem
#define predfile "data/predicate.dat" // File containing the default predicate

#define klimit 20

// Supporting functions :- 
int project(char *in, int set[], int l);
char applypredicate(char *in, int set[], char pred[], int l);
char applypolynomial(char *in, int set[], char poly[], int l);
char applyshamir(char *in, int set[], char pred[], int l);

// The main functions :-
// a) Function to use default setsystem and without any iterations
void eval(char *input, char *output);

// b) Function to use the default setsystem and predicate but with specified iterations
void evalIterative(char *input, char *output, int iter); 

// c) Function to use specified files for setsystem and predicate
void evalGivenFiles(char *input, char *gfile, char *pfile, char *output, int iter);

// d) Function to take setsystem and predicate from the memory as specified
//    by arguments. (MOST IMPORTANT USAGE)
void evalGiven(int n, int d, char *input, int collection[][klimit], char predicate[], char *output, 
		  int iter,  char (*apply)(char *, int [], char [], int));

// Fucntion to invoke OWF with a single iteration (default usage)
void eval(char *input, char *output) {
	 evalIterative(input, output, 1);
}

// Function to use the default setsystem and predicate with specified iterations
void evalIterative(char *input, char *output, int iter) {
	 evalGivenFiles(input, genfile, predfile, output, iter);
}

// The main function to get the output for a given bit-string
// Generalised for iterations 
void evalGivenFiles(char *input, char *gfile, char *pfile, char *output, int iter) {
	 // Read the values from "generated.dat"
	 FILE *fp;
	 int test1;
	 
	 test1 = (int)fopen(gfile, "r");
	 if (!test1) {
		  printf("\nThe file for setsystem could not be opened. Make sure you are in the right directory while executing this command.\n\n");
		  exit(1);
	 }
	 fp = fopen(gfile, "r");
	 
	 int n, d;
	 fscanf(fp, "%d\n%d\n", &n, &d);
	 
	 int collection[n][klimit];
	 int val, power;
	 char pred;
	 for (int k=0; k<n; k++) {
		  for (int j=0; j<d; j++) {
			   fscanf(fp, "%d\n", &val);
			   collection[k][j] = val;
		  }
	 }
	 fclose(fp);
     
	 int choice, d1;
	 
	 test1 = (int) fopen(pfile, "r");
	 if (!test1) {
		  printf("\nThe file for predicate could not be opened. Make sure you are in the right directory while executing this command.\n\n");
		  exit(1);
	 }
	 fp = fopen(pfile, "r");
	 
	 fscanf(fp, "%d\n%d\n", &choice, &d1);
	 
	 // Check if the 2 d's match
	 if (d1 != d) {
		  // For the case of the inner-product-predicate, d is not important
		  if (d != 2) {
			   printf("An error occurred while reading the stored expander and predicate. The value of d is not compatible.\n");
			   printf("Run 'gen' again.\n");
			   fclose(fp);
			   exit(1);
		  }
	 } else {
		  switch(choice) {
			   case 0: {
					power = (int) pow( 2.0, (double) d);
					char predicate[power];
					for (int k=0; k<power; k++) {
						 fscanf(fp, "%c\n", &pred);
						 predicate[k] = pred;
					}
					fclose(fp);
					evalGiven(n, d, input, collection, predicate, output, iter, applypredicate);
					break;
					   }
			   case 1: 
					{
					power = choose(d, 2);
					char polynomial[power];
					for (int k=0; k<power; k++) {
						 fscanf(fp, "%c\n", &pred);
						 polynomial[k] = pred;
					}
					fclose(fp);
					evalGiven(n, d, input, collection, polynomial, output, iter, applypolynomial);
					break;
					}
			   case 2:
					char dummy[1];
					evalGiven(n, d, input, collection, dummy, output, iter, applyshamir);
				    break;			
		  }
	 }
}

// Function to project the input string onto a particular 
// subset passed as an argument
int project(char *in, int set[], int l) {
	 int num = 0;
	 char currentbit;
	 for (int k=0; k<l; k++) {
		  currentbit = in[set[k]];
		  num *= 2;
		  if (currentbit == '1') {
			   num += 1;
		  }
	 }
	 return num;
}

// Function to evaluate the apply the random predicate (first option)
char applypredicate(char *in, int set[], char pred[], int l) {
	 return pred[project(in, set, l)];
}

// Function to apply the d-variant polynomial for the
// second option.
char applypolynomial(char *in, int set[], char poly[], int l){
	 int pointer = 0;
	 char currentbit1, currentbit2, currentcoeff, finalbit;
	 finalbit = '0';
	 for (int k=0; k<l; k++) {
		  for (int j=k+1; j<l; j++) {
			   currentbit1 = in[set[k]];
			   currentbit2 = in[set[j]];
			   currentcoeff = poly[pointer++];
			   if ((currentbit1 == '1') && (currentbit2 == '1') && (currentcoeff == '1')) {
					finalbit = (finalbit == '1')?'0':'1'; // Modulo 2 operation
			   }
		  }
	 }
	 return finalbit;
}

// Function to apply the inner product (third option)
// pred[] is a dummy argument
char applyshamir(char *in, int set[], char pred[], int l) {
	 char finalbit = '0';
	 for (int k=0; k<l/2; k++) {
		  if ((in[set[k]] == '1') && (in[set[k+(l/2)]] == '1'))
			   finalbit = (finalbit == '1')?'0':'1'; // Modulo 2 operation
	 }
	 if (l%2 != 0) {
		  // If l is odd, i just xor the last bit with the value computed till now.
		  // In practice, l will not be odd for shamir's case
		  finalbit = (finalbit == in[set[l-1]])?'0':'1';
	 }
	 return finalbit;
}

// Function to compute the output using the random predicate
void evalGiven(int n, int d, char *input, int collection[][klimit], char predicate[], 
		  	char *output, int iter, char (*apply)(char *, int [], char [], int)) {
	 int proj;
	 int itercount = 0;
	 char nextbit;
	 char intermediate[n];

	 for (int k=0; k<n; k++) {
		  nextbit = apply(input, collection[k], predicate, d);
		  output[k] = nextbit;	 
	 }
		  
	 while (++itercount < iter) {
		  for (int k=0; k<n; k++) {
			   nextbit = apply(output, collection[k], predicate, d);
			   intermediate[k] = nextbit;	 
		  }
		  copyarray_char(intermediate, output, n);
	 }
}

// Function to read the file named in command line i/p for running 
// the OWF/PRF :-
void readFile(FILE *fp, char *input, int n) {
	 char current;
	 for (int k=0; k<n; k++) {
		  current = getc(fp);
		  if (current == (char)EOF) {
			   printf("Error in input file. Too few characters.\n");
			   exit(1);
		  } else if ((current != '1') && (current != '0')) {
			   printf("Error in input file. It should contain only 1's and 0's");
		  	   exit(1);
		  } else {
			   input[k] = current;
		  }
	 }
}

