#!/usr/bin/perl -w
use strict;
use FileHandle;
my $FILE_CREATION_DIR = "sctvtmp";

my $tvppc_file_name = shift;
my $tvc_file_name   = shift;

local $::opts = {
    debug      => 0,
    queryfalse => 0,
    dontruncvc => 0,
    timecvc    => 0,
    printmaxp  => 0
};



sub printopts
{
    print STDERR "Supported options are: ";
    foreach my $o (keys %$::opts) {
        print STDERR "$o ";
    }
    print "\n";   
}

while ($_ = shift)
{
    if (defined $::opts->{$_})
    {
        $::opts->{$_} = 1;
    } else {
        print STDERR "Unsupported option '$_'!\n";
        printopts();
        die;
    }
}
defined $tvc_file_name and defined $tvppc_file_name or 
printopts() and die "Usage: $0 <tvppc_file_name> <tvc_file_name>";

if (not -e $FILE_CREATION_DIR) {
   mkdir $FILE_CREATION_DIR,0777;
}
if (not -d $FILE_CREATION_DIR) {
    die "Needed working directory '$FILE_CREATION_DIR' cannot be created,".
        " since a file of the same name already exists.";
}

my $tvcfile   = new FileHandle("<$tvc_file_name");
my $tvppcfile = new FileHandle("<$tvppc_file_name");
defined $tvcfile and defined $tvppcfile or 
die "Cannot open one of the input files";
####################################################
# Read the c_stdout file
####################################################
my $tvc_defs      = get_section($tvcfile,
                                "BEGIN-C-HEADER",
                                "END-C-HEADER");
if ($::opts->{debug}) {
    print "The TVC defs is:\n$tvc_defs";
    print "\n End of tvc_defs \n";
}
my $tvc_asserts   = get_exps($tvcfile,
                             "ASSERT",
                             "BEGIN-TVC-ASSERTS", 
                             "END-TVC-ASSERTS");
if ($::opts->{debug}) {
    foreach my $a (keys %$tvc_asserts)
    {
        print "ASSERT FOR TVC VAR ($a) is $tvc_asserts->{$a}\n";
    }
}
my $queries       = get_exps($tvcfile,
                             "QUERY", 
                             "BEGIN-QUERIES",
                             "END-QUERIES");

if ($::opts->{debug}) {
    foreach my $q (keys %$queries)
    {
        print "QUERY FOR VAR ($q) is $queries->{$q}\n";
    }
}
my $max_c_annotation_nums = get_max_annotation_nums($queries,"c");
if ($::opts->{debug}) {
	foreach my $mp (keys %$max_c_annotation_nums) 
	{
		print "MAX P FOR QUERY $mp is $max_c_annotation_nums->{$mp}\n";
	}
}

####################################################
# Read the a_stdout file
####################################################
my $common_header = get_section($tvppcfile,
                                "BEGIN-COMMON-HEADER",
                                "END-COMMON-HEADER");
if ($::opts->{debug}) {
    print "The common header is:\n$common_header";
}

my $asserts       = get_exps($tvppcfile, 
                             "ASSERT", 
                             "BEGIN-ASSERTS",
                             "END-ASSERTS");
if ($::opts->{debug}) {
    foreach my $a (keys %$asserts)
    {
        print "ASSERT FOR VAR ($a) is $asserts->{$a}\n";
    }
}

my $max_a_annotation_nums = get_max_annotation_nums($asserts,"a");
if ($::opts->{debug}) {
	foreach my $mp (keys %$max_a_annotation_nums) 
	{
		print "MAX P FOR $mp is $max_a_annotation_nums->{$mp}\n";
	}
}

my $unequalities  = get_section($tvppcfile,
                             "BEGIN-UNEQUALITIES",
                             "END-UNEQUALITIES");

my $name_mapping  = get_exps($tvppcfile, 
                             "ASSERT", 
                             "BEGIN-MAPPING",
                             "END-MAPPING");
if ($::opts->{debug}) {
    foreach my $n (keys %$name_mapping)
    {
        print "MAPPING FOR VAR ($n) is $name_mapping->{$n}\n";
    }
}

generate_query_files($queries, $common_header, $tvc_defs, $asserts,
                     $unequalities,
                     $name_mapping, 
                     $::opts->{queryfalse}, !$::opts->{dontruncvc}, 
                     $::opts->{timecvc}, 
                     $max_a_annotation_nums, $max_c_annotation_nums,
                     $tvc_asserts);
##############################
# For each variable, check the ASSERT value for variables of the type 'pXXX'
# and select the largest value XXX that appears.
##############################
sub get_max_annotation_nums
{
	my ($asserts,$assert_type) = @_;
	my $res = {};

	foreach my $a (keys %$asserts)
	{
		my $max_p = 0;
		my $exp = $asserts->{$a};
                if ($assert_type eq "a") {
                    while ($exp =~ /p(\d+)/gsm) {
			if ($1 > $max_p) {
                            $max_p = $1;
			}
                    }
                } elsif ($assert_type eq "c") {
                    while ($exp =~ /pc(\d+)/gsm) {
			if ($1 > $max_p) {
                            $max_p = $1;
			}
                    }
                } else {
                    die "*** Internal error!\n".
                        "*** get_max_annotation_nums called with bad ".
                            " assert_type = '$assert_type'";
                }
		$res->{$a} = $max_p;
	}
	return $res;
}

##############################
# Returns a dictionnary from the variable names to their queries.
##############################
sub get_exps
{
    my ($file, $exp_prefix, $begin_section_mark, $end_section_mark) = @_;
    my $exps = {};
    my $line;
    my $began = 0;
    my $exp = "";
    my $in_exp = 0;
    while ($line = <$file>)
    {
        if ($began == 0) {
            if ($line =~ /\Q$begin_section_mark/) {
                $began = 1;
            } else {
                # do nothing
            }
        } else {
            if ($line =~ /\Q$end_section_mark/) {
                $exp =~ /\Q$exp_prefix\E\s+(.*?)\s*=\s*(.*)/sm;
                my $var = $1;
                my $val = $2;
                chomp($val);
                $exps->{$var} = $val;
                last;
            } elsif ($in_exp) {
                if ($line =~ /^\Q$exp_prefix/) {
                    # get variable name
                    $exp =~ /\Q$exp_prefix\E\s+(.*?)\s*=\s*(.*)/sm;
                    my $var = $1;
                    my $val = $2;
                    chomp($val);
                    $exps->{$var} = $val;
                    $exp = $line;
                } else {
                    $exp .= $line;
                }
            } else {
                if ($line =~ /^\Q$exp_prefix\E/) {
                    $in_exp = 1;
                    $exp .= $line;
                } else {
                    # do nothing
                }
            }
        }
    }

    if ($in_exp) {
        $exp =~ /\Q$exp_prefix\E\s+(.*?)\s*=\s*(.*)/sm;
        my $var = $1;
        my $val = $2;
        chomp($val);
        $exps->{$var} = $val;
    }

    return $exps;
}

###########################
# this functions receives a filehandle a 'begin_section_mark' and an 
# 'end-section-mark' and returns the section that is inbetween these marks in
# the file
###########################
sub get_section
{
    my ($file, $begin_section_mark, $end_section_mark) = @_;
    my $common_header = "";
    my $in_header = 0;
    my $line;
    while ($line = <$file>)
    {
        if ($in_header) {
            if ($line =~ /\Q$end_section_mark/) 
            {
                last;
            } else {
                $common_header .= $line;
           }
        } else {
            if ($line =~ /\Q$begin_section_mark/) 
            {
                $in_header = 1;
            } else {
                # do nothing
            }
        }
    }
    return $common_header;
}

###############################
# This function creates the output files that will be checked by CVC.
# For each query variable, create an output file of the same name (with a 
# ".cvcin" extension), put the common header in it and add the relevant assert 
# (using the mapping).
# Also the annotation variables (p's) are added to every query.
# Finally, for every variable, a file with a ".queryfalse" is also created, in
# order to verify that cvc FAILS when it verifies it.
###############################
sub generate_query_files
{
    my ($query_vars, $common_header, $tvc_defs,
        $asserts, $unequalities, $name_mapping, 
        $create_query_false_files, $runcvc, $timecvc, 
	$max_a_annotation_nums, $max_c_annotation_nums,
        $tvc_asserts) = @_;

    my $validation_errors = 0;
    my $validation_true = 0;

    my $query_false_string = "QUERY FALSE;\n";

    # create the mappings_string
    my $mappings_string = "";
    my $assert_name = "";
    foreach my $mi (keys %$name_mapping)
    {
        $mappings_string .= "ASSERT $mi = $name_mapping->{$mi}\n";
        print "'$mi' MAPPED TO '$name_mapping->{$mi}'\n";
    }
    foreach my $q (keys %$query_vars)
    {
        # patch until local variables will not be queried:
        #if ($q !~ /^PC_X/) {
        if ($q !~ /^PMEM/) {
            next;
        }
        $assert_name = $name_mapping->{$q};
        if (not defined $assert_name) {
            #die "ERROR! No mapping is defined for $q.";
            print "*** ERROR ***\n";
            print "No mapping is defined for variable '$q'!\n";
            print "PROCEEDING TO NEXT VARIABLE.\n";
            next;
        }
        # remove trailing semi-colon
        $assert_name =~ s/;$//g;
        if (not defined $asserts->{$assert_name}) {
            #   die "ERROR! No assert is defined for $assert_name.";
            print "*** ERROR ***\n";
            print "No assert is defined for '$assert_name'!\n";
            print "PROCEEDING TO NEXT VARIABLE.\n";
            next;
        }
        
        # create the annotation_asserts_string
        my $annotation_asserts_string = "";
        for (my $i = 1; $i < $max_a_annotation_nums->{$assert_name} + 1; $i++)
        {
            my $pname = "p$i";
            $annotation_asserts_string .= 
                "ASSERT $pname = $asserts->{$pname}\n";
        }
        # create the tvc_annotation_asserts_string
        my $tvc_annotation_asserts_string = "";
        for (my $i = 1; $i < $max_c_annotation_nums->{$q} + 1; $i++)
        {
            my $pname = "pc$i";
            $tvc_annotation_asserts_string .= 
                "ASSERT $pname = $tvc_asserts->{$pname}\n";
        }
        print "Working on variable '$q' (='$assert_name')\n";
	if ($::opts->{printmaxp} == 1) {
	    print "Annotations generated up to: ".
		  "$max_a_annotation_nums->{$assert_name}\n";
	    print "TVC annotations generated up to: ".
		  "$max_c_annotation_nums->{$q}\n";
	}
        my $assert_string = "ASSERT $assert_name = $asserts->{$assert_name}\n";
        my $query_string = "QUERY $q = $query_vars->{$q}\n";

        my $output_file_name = "$FILE_CREATION_DIR/$q.cvcin";
        my $output_file = new FileHandle(">$output_file_name");
        defined $output_file or 
            die "Cannot open file $output_file_name for creation.";
        print $output_file "\n\%Common header\n";
        print $output_file $common_header;
        print $output_file "\n\%TVC defs (c header)\n";
        print $output_file $tvc_defs;
        print $output_file "\n\%Assembler annotations\n";
        print $output_file $annotation_asserts_string;
        print $output_file "\n\%C annotations\n";
        print $output_file $tvc_annotation_asserts_string;
        print $output_file "\n\%The assembler assert\n";
        print $output_file $assert_string;
        print $output_file "\n\%Variable mapping\n";
        print $output_file $mappings_string;
        print $output_file "\n\%Integer unequalities\n";
        print $output_file $unequalities;
        print $output_file "\n\%The query\n";
        print $output_file $query_string;
        undef $output_file;
        if ($runcvc) {
            my $cvcres = runcvc($output_file_name, $timecvc);
            $validation_true++;
            if ($cvcres != 1) {
                print "ERROR: Bad translation suspected for ";
                print "Variable: $q ($assert_name).\n";
                print "File: $output_file_name\n";
                print "CVC said the query was not valid.\n";
                $validation_errors++;
            }
        }

        if ($create_query_false_files) {
            $output_file_name = "$FILE_CREATION_DIR/$q.queryfalse";
            $output_file = new FileHandle(">$output_file_name");
            defined $output_file or 
                die "Cannot open file $output_file_name for creation.";
            print $output_file $common_header;
            print $output_file $annotation_asserts_string;
            print $output_file $tvc_annotation_asserts_string;
            print $output_file $assert_string;
            print $output_file $mappings_string;
            print $output_file $query_false_string;
            undef $output_file;

            if ($runcvc) {
                my $cvcres = runcvc($output_file_name, $timecvc);
                $validation_true++;
                if ($cvcres != 0) {
                    print "WARNING: Verification Error:\n";
                    print "The validation cannot be trusted since the ".
                        "assertions seem to contain an internal error:\n";
                    print "Variable: $q ($assert_name).\n";
                    print "File: $output_file_name\n";
                    print "CVC said that QUERY FALSE was valid.\n";
                    $validation_errors++;
                }
            }
        }
    }

    if ($runcvc) {
        if (($validation_errors == 0) && ($validation_true != 0)) {
            print "Translation valid.\n";
            print "\n";
            print '      _____        '."\n";
            print '     /     \       '."\n";
            print '    | O   O |      '."\n";
            print '    |   ^   |      '."\n";
            print '    |  \_/  |      '."\n";
            print '     \_____/       '."\n";
        } else {
            print "Translation invalid!!!\n";
            if ($validation_errors == 0) {
                print "The validation process was not completed due to unknown errors \n";
            } else {
                print "$validation_errors errors were found.\n";
            }
            print "\n";
            print '      _____        '."\n";
            print '     /     \       '."\n";
            print '    | X   X |      '."\n";
            print '    | ` o \' |      '."\n";
            print '    |   O   |      '."\n";
            print '     \_____/       '."\n";
        }
    }
}

# This function will run cvc on the filename it gets and return 0 if the file
# is invalid and 1 if it is valid.
sub runcvc
{
    my ($filename, $time) = @_;
    if ($time) {
        my $cvc_output = new FileHandle("time cvc < $filename|");
        my $last_line = <$cvc_output>;
        while (<$cvc_output>) {
            $last_line = $_;
        }
        print "$last_line\n";
        return 0;
    }
    print " $filename \n";
    my $cvc_output = new FileHandle("cvc < $filename|");
    while (<$cvc_output>)
    {
        if (/^Valid.$/) {
            # the following line was added, to avoid "broken pipe message
            while(<$cvc_output>) {} 
            close $cvc_output;
            return 1;
        } elsif (/^Invalid.$/) {
            while(<$cvc_output>) {}
            close $cvc_output;
            return 0; 
        } else {
            #die "ERROR: Unknown output from cvc: $_.";
		print "*** ERROR ***\n";
		print "Unknown error from cvc: $_\n";
                while(<$cvc_output>) {}
		close $cvc_output;
		return 0;
        }
    }
    #die "ERROR: No output gotten from cvc!";
	print "*** ERROR ***\n";
	print "No output gotten from cvc!\n";
	return 0;
}








